//
//  ToneGeneratorViewController.m
//  ToneGenerator
//
//  Created by Stanko Juzbašić 2014/01/19
//  Copyright 2014 Stanko Juzbašić. All rights reserved.
//  Ported, adapted and extended from iOS code by Matt Gallagher on 2010/10/20.
//  Copyright 2010 Matt Gallagher. All rights reserved.
//
//  Permission is given to use this source code file, free of charge, in any
//  project, commercial or otherwise, entirely at your risk, with the condition
//  that any redistribution (in part or whole) of source code must retain
//  this copyright and permission notice. Attribution in compiled projects is
//  appreciated but not required.
//
//  V. 1.0.8 fixed threading issues in GUI updating and alerts.

#import "ToneGeneratorViewController.h"

Boolean changed = FALSE;
Boolean changen = FALSE;
//_____________________________________________________________________________
# pragma mark - helpers -
//_____________________________________________________________________________
// generic error handler - adapted from "Learning Core Audio",
//         ISBN: 978-0-321-63684-3
// if err is nonzero, posts error message and exits program
//_____________________________________________________________________________
static void CheckError(OSStatus error, const char *operation)
{   

	if (error == noErr) return;
	
	char str[20];
    char cstr[265];
	// see if it appears to be a 4-char-code
	*(UInt32 *)(str + 1) = CFSwapInt32HostToBig(error);
	if (isprint(str[1]) && isprint(str[2]) && isprint(str[3]) && isprint(str[4])) {
		str[0] = str[5] = '\'';
		str[6] = '\0';
	} else
		// no, format as an integer
		sprintf(str, "%d", (int)error);
    
	fprintf(stderr, "Error: %s (%s)\n", operation, str);
    strcpy(cstr, operation);
    strcat(cstr, " (");
    strcat(cstr, str);
    strcat(cstr, ")\n");
    strcat(cstr, "Application will quit");
    
    NSString *theAlertMessage = [NSString stringWithUTF8String:cstr];
    
    NSString *ok = @"OK";
    //NSRunCriticalAlertPanel(@"Tone Generator: Hardware Error:",theAlertMessage, @"OK", nil, nil);
    
    __block BOOL killIt = FALSE;
    dispatch_async(dispatch_get_main_queue(), ^{
        killIt = (NSRunCriticalAlertPanel(@"Tone Generator: Hardware Error:",
                                          theAlertMessage,
                                          ok,
                                          nil,
                                          nil) == NSAlertDefaultReturn);        
    });
    do
    if (killIt == TRUE) exit(1);
    while(1);
    
	exit(1);
}
//_____________________________________________________________________________
double Frequency(double mPitch)
{
	return TuningA(A) * pow(2., (mPitch - 69. + bend*bendd) / 12.);
}
//_____________________________________________________________________________
double Pitch(double frequency)
{
    double f = frequency/TuningA(A);
    f = 12.*log2(f)+69;
    return f;
}
//_____________________________________________________________________________
double TuningA(double A)
{
	//return 440.0;
    return A;
}
//_____________________________________________________________________________
//converts lin value slider position to log frequency
double logfreq (double val, double min, double max)
{   
    
    return min*pow((max/min),(val-min)/(max-min));
}
//_____________________________________________________________________________
//converts log frequency to a lin value slider position
double logval (double val, double min, double max)
{
    return min + (max-min)*log(val/min)/log(max/min);
}
//_____________________________________________________________________________
// diode transfer function inspired by 
// the work of Richard Hoffmann Burchardi 
double da(double v)
{
    // Get the tone parameters out of the view controller
    
    //germanium, assuming signal betw. -.2 and .2 V
    
    if( v > 0.)
        return     pow(v, nonl);
    return -1.*pow(fabs(v), nonl);
    
}
//_____________________________________________________________________________
# pragma mark - MIDI -
//_____________________________________________________________________________
// monophonic controller with last held note priority.
// depending on manufacturers, we may run into any of the two "note-off" scenarios 
//      true note-off-message and zero-velovity-note-on-message
//      we have to check against both of them

void midiReadProc(const MIDIPacketList *packetList, 
                  void* readProcRefCon, 
                  void* srcConnRefCon)
{
    MIDIPacket *packet = (MIDIPacket*)packetList->packet;
    int  j;
    //    static int p[16];
    int count = packetList->numPackets;
    for (j=0; j<count; j++) {
        //printPacketInfo(packet);
        Byte midiStatus = packet->data[0]; 
        Byte midiCommand = midiStatus >> 4;
        //        if ((midiCommand == 0x08) ||(midiCommand == 0x09)) 
        //is a note already being held?
        midiCommand = midiCommand & 0x7F;
        //        if(midiCommand != 0xF)NSLog(@"MIDI CMD: %x\n", midiCommand);
        if(midiCommand == 0x9)//note on
        {
            Byte note = packet->data[1] & 0x7F; 
            Byte velocity = packet->data[2] & 0x7F;
            
            if (velocity == 0x0){ //hacked note-off!!!
                //hold--;
                if( hold > 0){//find which note has been released!
                    for(int i = 0; i <= hold; i++)                        
                        if(note == n[i]){
                            if (i == hold) {
                                goto h;
                            }                           
                            for (int j = i+1; j < hold; j++)
                                n[j-1] = n[j];
                        }
                h:    
                    hold--;
                    if(hold > 0)
                        note = n[hold-1];
                    pitch= (double)note;
                }
                
            }else{                //note on!!!
                //if(hold > 0)    
                n[hold] = note;
                pitch = (double)note;
                hold++;
            }
        }  
        /**/
        //proper note off
        if(midiCommand == 0x08)
        {    
            Byte note = packet->data[1] & 0x7F;
            if( hold > 0){//find which note has been released!
                for(int i = 0; i <= hold; i++)                        
                    if(note == n[i]){
                        if (i == hold) {
                            goto hh;
                        }                           
                        for (int j = i+1; j < hold; j++)
                            n[j-1] = n[j];
                    }
            hh:    
                hold--;
                if(hold > 0)
                    note = n[hold-1];
                pitch= (double)note;
            }
          if(0)NSLog(@"Note OFF , hold = %d\n", hold);
        }
        //printf("HOLD = %d\n", hold);
        
        //0xB -control change: "1011"
        if(midiCommand == 0xB){
            Byte ctrl = packet->data[1] & 0x7F;
            if(ctrl == (0x01 & 0x7F)){          //C01 -> mod. depth
                Byte val  = packet->data[2] & 0x7F;
                bmi = val;
                mmi = (double)val/127.   ;
                changed = TRUE;
            }  
            if (ctrl == (0x0B & 0x7F)){         //C11 -> nonlinearity   
                Byte val  = packet->data[2] & 0x7F;
                bmi = val;
                nonl = .125*(double)val/127. + 1 ;
                changen = TRUE;
            }      
        }
        //pitch bend: 0xE
        if(midiCommand == 0xE){
            Byte valm = packet->data[2] & 0x7F;
            Byte vall = packet->data[1] & 0x7F;
            bend = 128*(double)valm + (double)vall - 8192.;
            bend = bend/8192.;
        }
        
        //  if(midiCommand != 0xF)NSLog(@"Hold = %d\n", hold);
        
        packet = MIDIPacketNext(packet);    
    }
}

//_____________________________________________________________________________
void printPacketInfo(const MIDIPacket* packet)
{   
    char s[255];
    char s1[100];
    double timeinsec = packet->timeStamp / (double)1e9;
    sprintf(s,"%9.3lf\t", timeinsec);
    int i;
    for (i=0; i<packet->length; i++) {
        if (packet->data[i] < 0x7f) {
            sprintf(s1,"%d ", packet->data[i]);
        } else {
            printf(s1,"0x%x ", packet->data[i]);
        }
    }
    strcat(s, "\n");
    NSString *sNS = [NSString stringWithUTF8String:s];
    NSLog(@"%@",sNS);
    
}

//_____________________________________________________________________________
# pragma mark - render callback -
//_____________________________________________________________________________
//  The render callback generates blocks of inNumberFrames audio samples
//  Accumulated phase read from and stored back into the view controller
//  As control values are getting updated once each "inNumberFrames" 
//      the missing values must be interoplated and integrated,
//      else we hear the "zipper" effect when moving the controls
//_____________________________________________________________________________
OSStatus RenderTone(
	void *inRefCon, 
	AudioUnitRenderActionFlags 	*ioActionFlags, 
	const AudioTimeStamp 		*inTimeStamp, 
	UInt32 						inBusNumber, 
	UInt32 						inNumberFrames, 
	AudioBufferList 			*ioData)
{
	// Fixed amplitude so far
    Float32 holdme = ((hold+holdit) > 0 ? 1. : 0.);
    //holdme = 1.;
	const Float32 amplitude = 0.25;

    
    Float32 dph[inNumberFrames];
    Float32 amp[inNumberFrames];

	// Get the tone parameters out of the view controller
	ToneGeneratorViewController *viewController =
		(ToneGeneratorViewController *)inRefCon;
    
    viewController->amplitude = amplitude*holdme;
    
    NSNumber *pitchNumber = [[NSNumber alloc]initWithDouble:pitch];
    
    switch (viewController->midif) {
        case 1:
            
        //extract frequency from midi note:
        //viewController->frequency = ...

            viewController->frequency = Frequency(pitch);
            if(viewController->frequency != viewController->previous_frequency)
                dispatch_async(dispatch_get_main_queue(),
                               ^{
                                   [viewController midiIdFreqChanged: pitchNumber];
                               });
            //debug
            //NSLog(@"MIDI: %3.1lf %3.1lfHz\n", pitch, viewController->frequency);
            break;            
        default:
            viewController->amplitude = amplitude*holdit;
            break;
    }

    switch (viewController->state){
        case 1:
            for(UInt32 i = 0; i < inNumberFrames; i++)
                amp[i] = viewController->amplitude*i/(Float32)inNumberFrames;
            viewController->state = 2;
            break;
        case 2:
            for(UInt32 i = 0; i < inNumberFrames; i++)
                amp[i] = viewController->previous_amplitude*(inNumberFrames - i)/(Float32)inNumberFrames + viewController->amplitude*(i)/(Float32)inNumberFrames;
            break;
        case 3:
            for(UInt32 i = 0; i < inNumberFrames; i++)
                amp[i] = viewController->amplitude*(inNumberFrames - i)/(Float32)inNumberFrames;
            viewController->state = 0;
            break;
    }
    
    [pitchNumber release];
    
    double phase = viewController->phase;
    //debug
    //NSLog(@"%lf\t%lf\n", viewController->frequency, viewController->sampleRate);
    double phase_increment = viewController->TWO_PI_NORM * viewController->frequency ;
    
    //for preventing "zipper effect" we build a table of phase increment time derivatives
    double delta = phase_increment - viewController->previous_phase_increment;
    for (UInt32 i = 0; i < inNumberFrames; i++)
        dph[i] = delta*i/(Float32)inNumberFrames;
    
    //debug
    //NSLog(@"%lf\t%lf\t%lf\n",  phase, phase_increment,  delta);

	// Mono tone generator: we only need one buffer
	const int channel = 0;
   
	Float32 *buffer = (Float32 *)ioData->mBuffers[channel].mData;

	// Samples' generation, rendering "inNumberFrames" samples
	for (UInt32 frame = 0; frame < inNumberFrames; frame++) 
	{
        //buffer[frame] = 0.;
		buffer[frame] = da(sin(phase)) * amp[frame];
		
        // Phase increment integration
		phase = phase + viewController->previous_phase_increment + dph[frame];
        
		if (phase  > TWO_PI)
			phase -= TWO_PI;
        
	}
        

	// Store the phase and previous_phase_increment back into the view controller
	viewController->phase = phase;
    viewController->previous_phase_increment = phase_increment;
    viewController->previous_frequency = viewController->frequency;
    viewController->previous_amplitude = viewController->amplitude;
    
	return noErr;
}

//_____________________________________________________________________________
# pragma mark - interface methods -
//_____________________________________________________________________________

@implementation ToneGeneratorViewController

@synthesize frequencySlider;
@synthesize playButton;
@synthesize frequencyTextField;
@synthesize sourceButton;
@synthesize linlogButton;
@synthesize tuningStepper;
@synthesize segCtrlMult;
@synthesize centerButton;
@synthesize testButton;
@synthesize holdButton;

//_____________________________________________________________________________
- (void) createMIDI
{

    //create MIDI input and client - - - - - - - - - - - 
    midiclient = 0;
    
    CheckError(MIDIClientCreate(CFSTR("MIDI_Client"), 
                                NULL, 
                                /*midiClientNotifyRefCon*/NULL, 
                                &midiclient), 
               "MIDI Client Create Error\nRelaunch Application:");
    
    CheckError(MIDIInputPortCreate(midiclient, 
                                   CFSTR("MIDI_Input"), 
                                   midiReadProc, 
                                   NULL, 
                                   &midiin),
               "MIDI Port Create Error\nRelaunch Application:");    
   
//connect MIDI - - - - - - - - - - - - - - - - - - - 

ItemCount nSrcs = MIDIGetNumberOfSources();
NSLog(@"MIDI Sources: %ld", (long)nSrcs);
ItemCount iSrc;

//    if(0)
for (iSrc=0; iSrc<nSrcs; iSrc++) {
    MIDIEndpointRef src = MIDIGetSource(iSrc);
    MIDIPortConnectSource(midiin, src, NULL);       
}
}
//_____________________________________________________________________________
- (void) disposeMIDI
{
OSStatus status;
ItemCount nSrcs = MIDIGetNumberOfSources();
ItemCount iSrc;

if (nSrcs > 0)         
for (iSrc=0; iSrc<nSrcs; iSrc++) {
    MIDIEndpointRef src = MIDIGetSource(iSrc);
    status = MIDIPortDisconnectSource(midiin, src);
}        
MIDIPortDispose(midiin);
MIDIClientDispose(midiclient);     
midiclient = 0;
}
//_____________________________________________________________________________

- (void)createToneUnit
{
    UInt32 size = sizeof(Float64);
	// Configure the search parameters to find the default I/O unit
	// (kAudioUnitSubType_RemoteIO on iOS;
	// kAudioUnitSubType_DefaultOutput or
    // kAudioUnitSubType_HALOutput on Mac OS X)
	AudioComponentDescription audioCompDesc;
	audioCompDesc.componentType = kAudioUnitType_Output;
	audioCompDesc.componentSubType = kAudioUnitSubType_HALOutput;
	audioCompDesc.componentManufacturer = kAudioUnitManufacturer_Apple;
	audioCompDesc.componentFlags = 0;
	audioCompDesc.componentFlagsMask = 0;
	
	// Get the default playback output unit (with input capacity)
	AudioComponent defaultOutput = AudioComponentFindNext(NULL, &audioCompDesc);
	NSAssert(defaultOutput, @"Can't find default output");
	
	// Create a new unit based on this that we'll use for I/O
    CheckError(AudioComponentInstanceNew(defaultOutput, &toneUnit),
			   "Couldn't get audio unit instance"); 
    
	// Set tone rendering function on the unit
	AURenderCallbackStruct input;
	input.inputProc = RenderTone;
	input.inputProcRefCon = self;
    CheckError( AudioUnitSetProperty(toneUnit, 
                                     kAudioUnitProperty_SetRenderCallback, 
                                     kAudioUnitScope_Input,
                                     0, 
                                     &input, 
                                     sizeof(input)),
               "Error setting callback: ");
    
    CheckError(AudioUnitGetProperty(toneUnit,                         
                                    kAudioUnitProperty_SampleRate,
                                    kAudioUnitScope_Output,
                                    0,
                                    &R, 
                                    &size),
               "Couldn't get sample rate");         

	// Set the format to 32 bit, single channel, floating point, linear PCM
	const int four_bytes_per_float = 4;
	const int eight_bits_per_byte = 8;
	//AudioStreamBasicDescription streamFormat;
	streamFormat.mSampleRate = R;
	streamFormat.mFormatID = kAudioFormatLinearPCM;
	streamFormat.mFormatFlags =
		kAudioFormatFlagsNativeFloatPacked | kAudioFormatFlagIsNonInterleaved;
	streamFormat.mBytesPerPacket = four_bytes_per_float;
	streamFormat.mFramesPerPacket = 1;	
	streamFormat.mBytesPerFrame = four_bytes_per_float;		
	streamFormat.mChannelsPerFrame = 1;	
	streamFormat.mBitsPerChannel = four_bytes_per_float * eight_bits_per_byte;
	 
    CheckError(AudioUnitSetProperty (toneUnit,
		kAudioUnitProperty_StreamFormat,
		kAudioUnitScope_Input,
		0,
		&streamFormat,
		sizeof(AudioStreamBasicDescription)),
        "Couldn't set StreamFormat on input scope / bus 0");
                 
    //set "global" variables
    A  =  [tuningStepper doubleValue];
    ET =  A/(128*pow(8.,.75));
    bendd  =  2.; //24 = +- 2 oct
    hold   =  0;
    nonl   =  1.;
    if(pitch != pitch)//check for nan and initialize 
        pitch  =  69; 
    
    //set public variables
    self->mult       = .25*powf(2.,segCtrlMult.selectedSegment);
    self->sampleRate =  R;
    self->TWO_PI_NORM= TWO_PI/R;
    self->phase      = 0.;  
    self->logview    =  [linlogButton state];
    if(logview)
    self->frequency  =  self->mult*A*logfreq([frequencySlider doubleValue],
                    [frequencySlider minValue], [frequencySlider maxValue])/440.;
    else 
    self->frequency  =  self->mult*A*[frequencySlider doubleValue]/440.;
    self->midif      =  [sourceButton state];
}

//_____________________________________________________________________________
- (IBAction)togglePlay:(NSButton *)sender
{
	if (toneUnit)
	{   
        //signal the callback to gently fade down amplitude
        state = 3;
        while (state != 0);
        //when render callback has finished the playback stop, uninit, dispose...
		AudioOutputUnitStop(toneUnit);

		AudioUnitUninitialize(toneUnit);
		AudioComponentInstanceDispose(toneUnit);        
        
		toneUnit = nil;
		[sender setTitle:@"Play"];       
	}
	else
	{
		[self createToneUnit];
        if(!midi){
            [self createMIDI];
            midi = TRUE;
            }
		
		// Initialize the unit
        CheckError(AudioUnitInitialize(toneUnit),
                   "Error initializing unit:"); 

		// Start playback
        state = 1;
        CheckError(AudioOutputUnitStart(toneUnit),
                   "Error starting unit: ");

		[sender setTitle:@"Stop"];
	}
}

//_____________________________________________________________________________
- (IBAction)sourceButtonClicked:(NSButton *)sender 
{
    switch (midif){
    case 0:
        midif = 1;
        pitch = Pitch(frequency);
        [sourceButton setTitle:@"MIDI"];
        break;
    case 1:
        midif = 0;
        [sourceButton setTitle:@"Slider"];
        break;
    default:
        break;
    }   
}

//_____________________________________________________________________________
# pragma mark slider 
//_____________________________________________________________________________
//(reads frm. slider!)
- (IBAction)sliderChanged:(NSSlider *)sender
{   
    
	double val =  [sender doubleValue];
    double min =  [sender minValue];
    double max =  [sender maxValue];
    
    if (!toneUnit) {
        if(!mult)mult = 1.;
        if((A < 435) || (A > 445)) A = 440;
        if (logview) {
            val = logfreq(val, min, max);
            val = val*mult*TuningA(A)/440.;
        }else
            val = val*mult*TuningA(A)/440.;    
        NSString *df = [[NSString alloc] initWithFormat:@"%4.1lf Hz", val];
        [frequencyTextField setStringValue: df];
        [df release]; 
        return;
    }
    
    if (logview) {
        //find which frequency corresponds slider position?
        val = logfreq(val, min, max);
        frequency = val*mult*TuningA(A)/440.;
        ;
    }else
        frequency = val*mult*TuningA(A)/440.; 
    
    NSString *f = [[NSString alloc] initWithFormat:@"%4.1lf Hz", frequency];
    [frequencyTextField setStringValue: f];
    [f release];
}

//_____________________________________________________________________________
//(reads frm. slider!)
- (IBAction)tuningStepperClicked:(NSStepper *)sender 
{   
    static double val;
    double a = [sender doubleValue];
    
    if(!toneUnit){
        val = [frequencySlider doubleValue];
        if (!val)val = a;
        if (!mult) mult = 1.;
        if (logview) val = logfreq(val,frequencySlider.minValue,frequencySlider.maxValue);
        
        val *= mult*a/440.;
        NSString *df = [[NSString alloc] initWithFormat:@"%4.1lf Hz", val];
        [frequencyTextField setStringValue: df];
        [df release];
        A = a;
        frequency = val;        
        return;
    }
        
    frequency *= a/A;
    NSString *f = [[NSString alloc] initWithFormat:@"%4.1lf Hz", frequency];
    [frequencyTextField setStringValue: f];
    [f release]; 
    A = a;
         
    [self updateFrequencyControls];
}
//_____________________________________________________________________________
//(invokes method which reads from slider, and calculates frequency)
- (IBAction)segCtrlMultClicked:(NSSegmentedControl *)sender 
{
    double f = frequency;
    if(!midif)
    f = [frequencySlider doubleValue];
    
    if(!toneUnit){
        if(!frequency)
            frequency = 440.;
        if(!A)
            A = 440;
        if(!mult)
            mult = 1;
    }

    double lf = f;
    //double prev_mult = mult;
    
    NSInteger clickedSegment = [sender selectedSegment];
    
    switch (clickedSegment) {
        case 0:
            mult = .25;
            break;
        case 1:
            mult = .5;
            break;  
        case 2:
            mult = 1.;
            break; 
        case 3:
            mult = 2.;
            break;
        case 4:
            mult = 4.;
        default:
            break;
    }

    if(!midif){
    if(logview){
        lf = logfreq(lf,[frequencySlider minValue], [frequencySlider maxValue]);
        lf = lf*mult*A/440.;
        frequency = lf;
    }
    
    if(!logview){
    f = f*mult*A/440.;
    frequency = f;
    }
    }
    [self updateFrequencyControls];
  
}
//_____________________________________________________________________________
//(invokes method which writes to slider!)
- (IBAction)linlogButtonClicked:(NSButton *)sender 
{
    NSInteger bstate = [sender state];
    switch (bstate) {
            
        case 1:
            logview = TRUE;
            [sender setTitle:@"Log."];
            break;
            
        default:
            logview = FALSE;
            [sender setTitle:@"Lin."];
            break;
    }
    [self updateFrequencyControls]; 
}
//_____________________________________________________________________________
//(writes directly to textfield & slider!)

- (IBAction)centerButtonClicked:(NSButton *)sender 
{   
    if(!toneUnit)
        if((A < 435) || (A > 445)) A = 440; 
    
    double f = TuningA(A); 
    
    if (midif)
        pitch = 69;
    
    if (logview) {        
    double min =  [frequencySlider minValue];
    double max =  [frequencySlider maxValue];
    frequency  =  f; 
    mult       =  2;    
            f  =  logval((f/mult)*(440./TuningA(A)), min, max); 
    
    [segCtrlMult setSelectedSegment: 3];    
    [frequencySlider setDoubleValue: f];
    }else{
    mult = 1;
    [segCtrlMult setSelectedSegment: 2];    
    [frequencySlider setDoubleValue: (f/mult)*(440./TuningA(A))];
    frequency = f;    
    }
        
    NSString *fr = [[NSString alloc] initWithFormat:@"%4.1f Hz", frequency];
	[frequencyTextField setStringValue:fr];
    [fr release]; 
    
}
//_____________________________________________________________________________
// (writes to silder!)
- (void)midiIdFreqChanged:(NSNumber*)pitchNumber
{
    
    //get freq from midi controller!!
    pitch = [pitchNumber doubleValue];
    double f = Frequency(pitch);
    //NSLog(@"f = %lf\tfrequency = %lf", f, frequency);
    
    //don't update what hasn't changed!!
    //we already select update criteria from RenderTone()!
    if(1)
    {
        frequency = f;
        double val =  frequency;
        double min =  [frequencySlider minValue];
        double max =  [frequencySlider maxValue];
        if (logview) {
            double lv  =  logval(val*440/(A*mult), min, max);
            if(lv > max){
                [segCtrlMult setSelectedSegment:segCtrlMult.selectedSegment + 1];
                mult *= 2.;
            }
            if(lv < min){
                [segCtrlMult setSelectedSegment:segCtrlMult.selectedSegment - 1];
                mult *= .5;
            }
            [frequencySlider setDoubleValue: logval(val*440/(A*mult), min, max)];
            // NSLog(@"LOGVIEW\n");
        }
        else{
            if (frequency*440/(A*mult) > max){
                [segCtrlMult setSelectedSegment:segCtrlMult.selectedSegment + 1];
                mult *= 2.;
            }
            if (frequency*440/(A*mult) < min){
                [segCtrlMult setSelectedSegment:segCtrlMult.selectedSegment - 1];
                mult *= .5;
            }
            
            [frequencySlider setDoubleValue:frequency*440/(A*mult)];
        }
        
        NSString *f = [[NSString alloc] initWithFormat:@"%4.1lf Hz", frequency];
        [frequencyTextField setStringValue: f];
        [f release]; 
    }
}
//_____________________________________________________________________________
// (writes to silder!)
- (void)midiFreqChanged
{
    frequency = Frequency(pitch);//get freq from midi controller!!

    double val =  frequency;
    double min =  [frequencySlider minValue];
    double max =  [frequencySlider maxValue];
  if (logview) {     
    double lv  =  logval(val*440/(A*mult), min, max);
      if(lv > max){
          [segCtrlMult setSelectedSegment:segCtrlMult.selectedSegment + 1];
           mult *= 2.;          
      }
      if(lv < min){
          [segCtrlMult setSelectedSegment:segCtrlMult.selectedSegment - 1];
          mult *= .5;          
      }      
    [frequencySlider setDoubleValue: logval(val*440/(A*mult), min, max)]; 
    }
   else{
       if (frequency*440/(A*mult) > max){
           [segCtrlMult setSelectedSegment:segCtrlMult.selectedSegment + 1];
           mult *= 2.;
       }
       if (frequency*440/(A*mult) < min){
           [segCtrlMult setSelectedSegment:segCtrlMult.selectedSegment - 1];
           mult *= .5;
       }

    [frequencySlider setDoubleValue:frequency*440/(A*mult)];        
   }
    
    NSString *f = [[NSString alloc] initWithFormat:@"%4.1lf Hz", frequency];
    [frequencyTextField setStringValue: f];
    [f release];          
}
//_____________________________________________________________________________
// (writes to silder!)
- (void)updateFrequencyControls
{   
    if (!toneUnit){
        if (!mult) mult = 1;
        if (!A)       A = 440;
        if (!frequency)frequency = A;
    }
    
    double min =  [frequencySlider minValue];
    double max =  [frequencySlider maxValue];    
    double val =  frequency; //(of the oscillator!!)
    
    //logarithmic view
    if (logview) {        
    double lv  =  logval(val*440/(A*mult), min, max);
    //also check if position exceeds boundaries
    [frequencySlider setDoubleValue: lv];   
   
    }else{
    
    //linear view
    if (frequency/mult < min){
        mult *= .5;
        [segCtrlMult setSelectedSegment: segCtrlMult.selectedSegment -1];
    }
    if (frequency/mult > max){
        mult *= 2;
        [segCtrlMult setSelectedSegment: segCtrlMult.selectedSegment +1];
    }    
    [frequencySlider setDoubleValue:(frequency/mult)*(440./TuningA(A))];
    }

    NSString *f = [[NSString alloc] initWithFormat:@"%4.1lf Hz", val];
    [frequencyTextField setStringValue: f];
    [f release];    
    return;
}
//_____________________________________________________________________________
# pragma mark end_slider 
//_____________________________________________________________________________
- (IBAction)testButtonClicked:(NSButton *)sender 
{
    NSLog(@"\nA = %4.1lf\nPitch = %3.0lf\nfrequency = %7.2lf\nmidif = %ld\nlogview = %d\n\
mult = %4.2f\n\n", A, pitch, self->frequency, self->midif, self->logview, self->mult);
}
//_____________________________________________________________________________
- (IBAction)holdButtonPressed:(NSButton *)sender {
    holdit = !holdit;
}
//_____________________________________________________________________________
- (void)stop
{
	if (toneUnit)
	{
		[self togglePlay:playButton];
	}
}
//_____________________________________________________________________________

@end
